function [centroid_data] = find_centroid_smart(varargin)
%This is a high order script that allows you to run find_centroidvstk.
%Workes basically like find_centroid_multi, but uses algorithms to pick the
%threshold, instead of using a hard and fast threshold.
%Synatax:   [] = find_centroid_smart('dirname',array,'theshold',0.5,'type',2,'type2',1,'norm',1)
%Input:     'dirname' = directory where the images are
%           'threshold' = the test thresholds, a range in which the
%               function will attempt to find a threshold.  The default
%               is: 0.03:0.01:0.17
%           type = 1(centroid), 2 (WeightedCentroid) or 0(local max), 
%               default = 0(local max)
%           type2 = 1(Area), 2(Max Intensity), 3(Min Intensity, 4(Pixel #)
%                   default = 0(Mean Intensity) Note: only works with
%                   centroids
%           norm = normalize the images, essentially stretching the image
%                   data over the full bit range.  Only works in centroid
%                   mode.  Default = 1, or on, and will not affect the
%                   intensity  measurements, now matches threshold.
%           object = out put the images of the objects.  Default = 0(off)
%               Does not work with local max
%           zfilter = requires the objects to be in more than 1 z slice
%           Note: as this is setup currently, if there is a directory with
%           images that also contain directories, those images will not be
%           processed.  Images are only processed in end directories.
%           limit = the lower and upper bounds of thresholds one will
%               consider.
%Output:    centroid_data = summary data of the centroids generated
%               formated such that data fields are seperated by animal type
%               and brain region.  Plus, all of the threshold data, and
%               file locations for each data point.

[thresh,dirname,type,type2,norm,object,zfilter,wavgroups] = parse(varargin);

%now grab those directory names
dir_struct = dir(dirname);  %grab the directory information
idx = [dir_struct.isdir];   %grab all of the isdir numbers
idx(1:2) = 0;               %remove the first two directories
dirnames = {dir_struct.name};   %grab the all of the names in the root
dirnames = dirnames(idx);  %don't want the first two
dir_tmp = repmat([dirname filesep],size(dirnames'));    %replicated the directory name for appending

%initiate
try
    load([dirname,filesep,'centroid_processing_status','.mat']);
    %now process the centroid data info, to make sure this is the same
    %continued process
    for k = 1:size(centroid_data_tmp,2)
        %first find if the processes matches any previous ran processes
        if centroid_data_tmp(k).norm==norm && centroid_data_tmp(k).object==object && centroid_data_tmp(k).zfilter==zfilter && centroid_data_tmp(k).type==type && min(centroid_data_tmp(k).type2==type2)
            itr = k;  %found it
            centroid_data = centroid_data_tmp(k).centroid_info;
            proc_status = centroid_data_tmp(k).proc_status;
            virgin = 0;
            break
        else  %start new
            itr = k+1;
            proc_status = [];
            centroid_data = [];
            virgin = 1;
        end
    end
catch
    centroid_data = [];
    proc_status = [];
    itr = [];  %The iteration counter is for keeping track of how many different centroid calculations one is running concurrently.
    virgin = 1;
end
if itr==1,itr = [];end  %this is strictly here for naming of directory.

warning('OFF','MATLAB:MKDIR:DirectoryExists');  %turn off this specific warning
%prepare the output directory
sep_loc = strfind(dirname,filesep);
out_tmp = [dirname(1:sep_loc(end)),'CalculatedCentroids',dirname(sep_loc(end):end),num2str(itr)];
mkdir(dirname(1:sep_loc(end)),'CalculatedCentroids')  %make the output root directories
mkdir([dirname(1:sep_loc(end)),'CalculatedCentroids'],[dirname(sep_loc(end)+1:end),num2str(itr)]);
out_tmp = repmat([out_tmp filesep],size(dirnames'));    %replicated the out_put name for appending
out_list = horzcat(cellstr(out_tmp),dirnames');      %now each cell contains the full output directory path

dirnames = cellstr([dir_tmp char(dirnames')]);      %now each cell contains the full directory path

if isempty(itr),itr = 1;end  %this is strictly here for naming of directory, so that the first directory does not have 1 behind it.

if ~isempty(dirnames)   %if the root only have files, then no need to go further
    while 1     %we are going to dig until all sub_dirs are found
        dir_tmp = [];   %clear!
        dir_end = 0;    %reset
        out_tmp = [];
        %level in
        for i = 1:size(dirnames,1)  %for each dir_step down one
            warning('OFF','MATLAB:MKDIR:DirectoryExists');  %turn off this specific warning
            mkdir(out_list{i,1});     %copy over this directory
            curr_out = [out_list{i,1} out_list{i,2}];     %current output directory
            curr_dir = dirnames{i};     %current directory
            dir_info = dir(curr_dir); %grab the current dir
            idx = [dir_info.isdir];   %grab all of the isdir numbers
            idx(1:2) = 0;   %toss the first two
            curr_names = {dir_info.name};   %grab the all of the 
            dir_nxt = curr_names(idx);    %the directories
            if isempty(dir_nxt)   %bottom
                dir_end = dir_end+1;    %one path to its conclusion
                %now lets see if this directory have the image files we
                %are looking for
                idx = [dir_info.isdir];   %grab all of the isdir numbers
                filenames = curr_names(~idx);   %files
                tifsrc = strfind(filenames,'.tif');     %serch for tif files
                if size(cell2mat(tifsrc),2)>=2    %if there are at least two images
                    %check if this directory has been completed already
                    if ~isempty(proc_status) && virgin~=1
                        for l = 1:size(proc_status,1)
                            if strcmpi(curr_dir,proc_status{l,1})  %if find match
                                forward = 0;  %done
                                break
                            else
                                forward = 1;  %do this
                            end
                        end
                    else  %do
                        forward = 1;
                    end
                    if forward
                        [wavgroups] = groupwav(dirnames);
                        disp(curr_out);  %print current directory to command line
                        %first parse the directory structure
                        dir_idx = find(curr_dir==filesep);  %where are all of slashes
                        ch_name = curr_dir(dir_idx(end)+1:end);     %grab the channel protein name, e.g., PSD95
                        reg_name = curr_dir(dir_idx(end-1)+1:dir_idx(end)-1);   %the tissue region, e.g., Layer5
                        ani_name = curr_dir(dir_idx(end-2)+1:dir_idx(end-1)-1); %the animal type, e.g, KO, WT, Wake, etc...
                        exp_name = curr_dir(dir_idx(end-3)+1:dir_idx(end-2)-1); %the experiment name, e.g., 20140211
                        %parse the ch_name further for consumption
                        spc_loc = isspace(ch_name);  %get the space locations for the name string
                        if sum(spc_loc)==2  %we want the specific formate '1 PSD95 594'
                            str_idx = find(spc_loc==1);
                            ch_name = ch_name(str_idx(1)+1:str_idx(2)-1);
                        end
                        %otherwise the ch_name is not modified
                        %now get the image stack
                        stk = openstk(curr_dir);
                        %lets figure out the threshold, smartly...
                        [threshold,category] = find_threshold(ch_name,curr_dir,type,type2,object,norm,zfilter,thresh,stk,wavgroups);
                        %run find_centroidvstk
                        [pivots_tmp,output_dir] = find_centroidvstk_loco('dirname',curr_dir,'type',type,'type2',type2,'object',object,...
                            'threshold',threshold,'norm',norm,'zfilter',zfilter,'dirout',out_list{i,1},'stk',stk);
                        %now lets store the centroids data for later retrieval
                        %now generate output data structure
                        if ~isempty(centroid_data)
                            ani_fields = fields(centroid_data);
                            if max(strcmpi(ani_fields,ani_name))
                                reg_fields = fields(centroid_data.(ani_name));
                            end
                            %first time through lets prepare the centroid data
                            %temp file
                            centroid_data_tmp(itr).norm = norm;  %save out some parameters as well
                            centroid_data_tmp(itr).object = object;
                            centroid_data_tmp(itr).zfilter = zfilter;
                            centroid_data_tmp(itr).type = type;
                            centroid_data_tmp(itr).type2 = type2;
                        end
                        if isempty(centroid_data)||~max(strcmpi(ani_fields,ani_name))||~max(strcmpi(reg_fields,reg_name))  %initiate if this is the first point
                            centroid_data.(ani_name).(reg_name).centroid = {[],ch_name;exp_name,size(pivots_tmp{1},1)};        %centroid number
                            centroid_data.(ani_name).(reg_name).threshold = {[],ch_name;exp_name,threshold};                   %threshold used
                            centroid_data.(ani_name).(reg_name).category = {[],ch_name;exp_name,category};                   %threshold method
                            centroid_data.(ani_name).(reg_name).location = {[],ch_name;exp_name,{curr_dir,out_list{i,1}}};     %location of files
                        else  %if not the first time lets first find where to place it
                            [x,y] = locate_xy(centroid_data.(ani_name).(reg_name).centroid,ch_name,exp_name);  %where should we put the data
                            %put in the labels
                            centroid_data.(ani_name).(reg_name).centroid{y,1} = exp_name;        %centroid number
                            centroid_data.(ani_name).(reg_name).threshold{y,1} = exp_name;                   %threshold used
                            centroid_data.(ani_name).(reg_name).category{y,1} = exp_name;                   %threshold method
                            centroid_data.(ani_name).(reg_name).location{y,1} = exp_name;     %location of files
                            centroid_data.(ani_name).(reg_name).centroid{1,x} = ch_name;        %centroid number
                            centroid_data.(ani_name).(reg_name).threshold{1,x} = ch_name;                   %threshold used
                            centroid_data.(ani_name).(reg_name).category{1,x} = ch_name;                   %threshold method
                            centroid_data.(ani_name).(reg_name).location{1,x} = ch_name;     %location of files
                            %put the data in
                            centroid_data.(ani_name).(reg_name).centroid{y,x} = size(pivots_tmp{1},1);        %centroid number
                            centroid_data.(ani_name).(reg_name).threshold{y,x} = threshold;                   %threshold used
                            centroid_data.(ani_name).(reg_name).category{y,x} = category;                   %threshold method
                            centroid_data.(ani_name).(reg_name).location{y,x} = {curr_dir,out_list{i,1}};     %location of files
                        end
                        %save out a temporary file
                        %save out status update, so next iteration will not
                        %have to redo a whole bunch of files
                        proc_status = vertcat(proc_status,{curr_dir,output_dir});
                        centroid_data_tmp(itr).centroid_info = centroid_data;
                        centroid_data_tmp(itr).proc_status = proc_status;
                        save([dirname,filesep,'centroid_processing_status','.mat'],'centroid_data_tmp');
                    end
                end
            else    %not the bottom
                curr_dir = repmat([curr_dir filesep],size(dir_nxt'));    %replicated the directory name for appending
                dir_tmp = vertcat(dir_tmp,cellstr([curr_dir char(dir_nxt')]));      %now each cell contains the full directory path
                curr_out = repmat([curr_out filesep],size(dir_nxt'));    %replicated the out_put name for appending
                out_tmp = vertcat(out_tmp,horzcat(cellstr(curr_out),dir_nxt'));      %now each cell contains the full output directory path
            end
        end
        if dir_end==size(dirnames,1)    %all directories are at an end
            break
        else    %move next level down
            dirnames = dir_tmp;
            out_list = out_tmp;
        end
    end
    %done save out the data
    centroid_data.norm = norm;  %save out some parameters as well
    centroid_data.object = object;
    centroid_data.zfilter = zfilter;
    centroid_data.type = type;
    centroid_data.type2 = type2;
    %now save it
    save([dirname,filesep,'centroid_data',datestr(now,'yyyymmdd.HHMM'),'.mat'],'centroid_data');
end

%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [thresh,dirname,type,type2,norm,object,zfilter,wavgroups] = parse(input)

thresh = 0.03:0.01:0.15;    
type = 0;
type2 = 0;      %default is 0; now you can define the accessory data, acquired instead of Luminance
norm = 1;
object = 0;
dirname = [];
zfilter = 0;
wavgroups = [];

%Parse the input
if ~isempty(input)
    for i = 1:2:size(input,2)
        if ischar(input{1,i});
            switch input{1,i}
                case 'dirname'
                    dirname = input{1,i+1};
                case 'threshold'
                    thresh = input{1,i+1};
                case 'type'
                    type = input{1,i+1};
                case 'type2'
                    type2 = input{1,i+1};
                case 'norm'
                    norm = input{1,i+1};
                case 'object'
                    object = input{1,i+1};
                case 'zfilter'
                    zfilter = input{1,i+1};
                case 'wavgroups'
                    wavgroups = input{1,i+1};
                otherwise
                    warning(['Your input ',input{1,i},' is not recognized.']);
            end
        else
            error(['The parameters you entered is incorrect.  Please check help.']);
        end
    end
end

%Get the location of the images you want to open
if isempty(dirname)
    dirname = uigetdir2('','Directory where your images are');    %get the directory
end
%--------------------------------------------------------------------------
%subfunction to locate where in the matrix we should put the data
function [x,y] = locate_xy(data_mat,xlabel,ylabel)
y = [];  %initiate
x = [];
x_size = size(data_mat,2);  %number of x items
y_size = size(data_mat,1);  %number of y items
for i = 1:y_size
    if strcmpi(ylabel,data_mat{i,1})  %if we find it then go find x
        y = i;
        break
    end
end
for j = 1:x_size
    if strcmpi(xlabel,data_mat{1,j})  %if we find it then return
        x = j;
        break
    end
end
if isempty(y)
    y = y_size+1;  %attach the data at the end
end
if isempty(x)  %if we don't find anything
    x = x_size+1;
end
%--------------------------------------------------------------------------
function [stk] = openstk(dirname)
%grab the filenames in the directory
dir_struct = dir(dirname);  %grab the directory information
idx = [dir_struct.isdir];   %grab all of the isdir numbers
filenames = {dir_struct.name};   %grab the all of the names in the root
filenames = filenames(~idx);  %don't want the directory names

%Put the virtual stack into an actual volume
[filenames] = findimgs(filenames);   %make sure we only take the image names
info = imfinfo([dirname,filesep,filenames{1}]);     %grab the file info
switch info.BitDepth    %what is the bit depth
    case 8
        imgclass = 'uint8';
    case 16
        imgclass = 'uint16';
    case 32
        imgclass = 'uint32';
    otherwise
        imgclass = 'double';
end

stk = zeros(info.Height,info.Width,size(filenames,2),imgclass); %generate holder variable

h = waitbar(0,'Opening Image: ','position',[10 50 275 50]);     %initiate progress bar
for i = 1:size(filenames,2)
    waitbar(i/size(filenames,2),h,['Opening Image: ',filenames{i}]);   %update progress
    stk(:,:,i) = imread([dirname,filesep,filenames{i}]);  %grab the image
end
close(h)

%--------------------------------------------------------------------------
%subfunction to find the threshold
function [threshold,category] = find_threshold(ch_name,curr_dir,type,type2,object,norm,zfilter,thresh,stk,wavgroups)

%initiate
% threshold = [];
category = {};  %how the threshold was determined.
% %parse limits,  limits are used if you just want to find thresholds in a
% %segemnet of the tested thresholds
% try
%     lower_lim = find(single(thresh)==single(limit(1)));
%     upper_lim = find(single(thresh)==single(limit(2)));
%     if isempty(lower_lim)  %if the limit is not the the threshold tested, find the nearest
%         lower_lim = find(thresh<limit(1),1,'last');
%     end
%     if isempty(upper_lim)  %if the limit is not the the threshold tested, find the nearest
%         upper_lim = find(thresh>limit(2),1,'first');
%     end
% catch  %no limit stated
%     lower_lim = 1;
%     upper_lim = size(thresh,2);
% end
% %get the actual allowed threshold range
% thresh_range = thresh(lower_lim:upper_lim);

%Lets collect our data first
[stat_data,num_data] = find_centroid_testrange_loco(curr_dir,type,type2,object,...
            thresh,norm,zfilter,stk,wavgroups);
%tdata (x,:) x = mean, median, mode
%now lets figure out the threshold
switch ch_name  %for some special channels we might have special rules
    %case {'PSD','psd','PSD95','psd95','PSD-95','psd-95'}
    otherwise  %the generic case
        %Calculate the lower bound first using correlations: seems
        %incredibly robust
        data_tmp = smooth(stat_data(7,:),'lowess');
        data_tmp = cal_curve(data_tmp',1,1);
        lower_bound_idx = peak_pick(data_tmp);
        lower_bound = thresh(lower_bound_idx);
        weight = 1;
        category = horzcat(category,['default: ',num2str(lower_bound)]);
        %Threshold at peak of centroid number, FYI used as a limit.  We
        %want the threshold to be ~near the peak of the most centroids
        %adjusted by measurments of noise
        thresh1_idx = peak_pick(num_data);
        if isempty(thresh1_idx)
            thresh1 = [];
            weight1 = [];
        else
            thresh1 = thresh(thresh1_idx);
            if thresh1<lower_bound
                thresh1 = [];
                weight1 = [];
            else
                weight1 = 1;
                category = horzcat(category,['centroid peak: ',num2str(thresh1)]);
            end
        end
        
        %Calculate correlation against another wavelength.  This is to
        %minimize blead through of other channel, if data supports it.
        data_tmp = smooth(stat_data(8,:),'lowess');
        thresh2_idx = nadir_pick(data_tmp');
        if isempty(thresh2_idx)
            thresh2 = [];
            weight2 = [];
        else
            thresh2 = thresh(thresh2_idx);
            if thresh2<lower_bound
                thresh2 = [];
                weight2 = [];
            else
                weight2 = 1;
                category = horzcat(category,['cross wavelength: ',num2str(thresh2)]);
            end
        end
        
        %Calculate the volume mean max as a measure of increased background
        %pickup
        data_tmp = smooth(stat_data(1,:),'lowess');
        data_tmp = cal_curve(data_tmp',1,1);
        thresh3_idx = peak_pick(data_tmp);
        if isempty(thresh3_idx)
            thresh3 = [];
            weight3 = [];
        else
            thresh3 = thresh(thresh3_idx);
            if thresh3<lower_bound
                thresh3 = [];
                weight3 = [];
            else
                weight3 = 0.5;
                category = horzcat(category,['volume mean: ',num2str(thresh3)]);
            end
        end
        
        
        %Calculate the threshold
        threshold = sum([lower_bound*weight,thresh1*weight1,thresh2*weight2,thresh3*weight3])/sum([weight,weight1,weight2,weight3]);

%         %Threshold at peak of curvature of centroid number.
%         k = cal_curve(num_data);
%         [cnum_idx] = peak_pick(k(lower_lim-1:upper_lim-1),num_max_idx);  %the curvature shifts the curve left by one.
%         cnum_thresh = thresh_range(cnum_idx);
%         if ~isempty(cnum_thresh)  %if not empty enter this in the categorization section
%             category = horzcat(category,'cnum');
%         end
%         %Threshold at peak of curvature of centroid number - smoothed.
%         k = smooth(k,'lowess')';
%         [scnum_idx] = peak_pick(k(lower_lim-1:upper_lim-1),num_max_idx);  %the curvature shifts the curve left by one.
%         scnum_thresh = thresh_range(scnum_idx);
%         if ~isempty(scnum_thresh)  %if not empty enter this in the categorization section
%             category = horzcat(category,'scnum');
%         end
%         %Threshold at peak of curvature in mean centroid volume. 
%         k = cal_curve(stat_data(1,:));
%         [cmean_idx] = peak_pick(k(lower_lim-1:upper_lim-1),num_max_idx);  %the curvature shifts the curve left by one.
%         cmean_thresh = thresh_range(cmean_idx);
%         if ~isempty(cmean_thresh)  %if not empty enter this in the categorization section
%             category = horzcat(category,'cmean');
%         end
%         %Threshold at peak of curvature in mean centroid volume - smoothed.
%         k = smooth(k,'lowess')';
%         [scmean_idx] = peak_pick(k(lower_lim-1:upper_lim-1),num_max_idx);  %the curvature shifts the curve left by one.
%         scmean_thresh = thresh_range(scmean_idx);
%         if ~isempty(scmean_thresh)  %if not empty enter this in the categorization section
%             category = horzcat(category,'scmean');
%         end

%         %Take all of the metrics together and generate a threshold
%         %combine the centroid number and mean curvature and smoothed
%         %curvature so we don't over weigh the two
%         cnum_thresh = nanmean([cnum_thresh,scnum_thresh]);
%         cmean_thresh = nanmean([cmean_thresh,scmean_thresh]);
%         %now combine all of the thresholds to generate a final threshold
%         threshold = nanmean([cnum_thresh,cmean_thresh,median_thresh,mode_thresh]);
%         %if no metrics fit, default to 0.1
%         if isempty(threshold)||isnan(threshold)
%             %if peak is above 0.1 threshold, use the peak
%             if thresh(num_max_idx)>0.1
%                 if num_max_idx>upper_lim  %check for limit
%                     num_max_idx = upper_lim;
%                 end
%                 threshold = thresh(num_max_idx);
%             else
%                 threshold = 0.1;
%             end
%             category = {'default'};
%         end
end
%--------------------------------------------------------------------------
%subfunction to pick the peak of a curve according to specific rules
function [peak_idx] = peak_pick(data)
%initialize
data_sz = size(data,2);  %grab the x size of the data
max_idx = find(data==max(data));     %locaiton of the max

if isempty(max_idx)
    peak_idx = [];
else
    if size(max_idx,2)==1  %move forward there is a single max
        peak_idx = max_idx;
    else  %multiple maxes lets check
        if max_idx(1)-max_idx(end)>size(max_idx,2)*0.3  %if the points are spread across a distance over 30% of the scale, dump them
            peak_idx = [];
        else  %otherwise take the lowest one
            peak_idx = max_idx(1);
        end
    end
    %if it is at the ends don't use
    if ~isempty(peak_idx)
        if peak_idx==1||peak_idx==data_sz
            peak_idx = [];
        end
    end
end

%--------------------------------------------------------------------------
%subfunction to pick the peak of a curve according to specific rules
function [nadir_idx] = nadir_pick(data)
%initialize
data_sz = size(data,2);  %grab the x size of the data
min_idx = find(data==min(data));     %locaiton of the max

if isempty(min_idx)
    nadir_idx = [];
else
    if size(min_idx,2)==1  %move forward there is a single max
        nadir_idx = min_idx;
    else  %multiple maxes lets check
        if min_idx(1)-min_idx(end)>size(min_idx,2)*0.3  %if the points are spread across a distance over 30% of the scale, dump them
            nadir_idx = [];
        else  %otherwise take the lowest one
            nadir_idx = min_idx(1);
        end
    end
    %if it is at the ends don't use
    if ~isempty(nadir_idx)
        if nadir_idx==1||nadir_idx==data_sz
            nadir_idx = [];
        end
    end
end
%--------------------------------------------------------------------------
%subfunction to calculate the curvature of a line

%the curvature of a circle drawn through three points is simply four times
%the area of the triangle formed by the three points divided by the product
%of its three sides. Using the coordinates of the points this is given by:

%K = 2*abs((x2-x1).*(y3-y1)-(x3-x1).*(y2-y1)) ./ ...
%  sqrt(((x2-x1).^2+(y2-y1).^2)*((x3-x1).^2+(y3-y1).^2)*((x3-x2).^2+(y3-y2).^2));

function [curvature] = cal_curve(line_data,absolute,rez)

if nargin<2
    absolute = 1;  %default turns absolute on
    rez = 1;
elseif nargin<3
    rez = 1;
end

%initiate x is always 1 2 3
x1 = 1;
x2 = 2;
x3 = 3;

%iterate through the line
for i = 1:size(line_data,2)
    y2 = line_data(i);
    if i<=rez
        y1 = nan;
    else
        y1 = line_data(i-rez);
    end
    if i+rez>size(line_data,2)
        y3 = nan;
    else
        y3 = line_data(i+rez);
    end
    switch absolute
        case 1  %absolute on
            curvature(i) = 2*abs((x2-x1).*(y3-y1)-(x3-x1).*(y2-y1)) ./ ...
                sqrt(((x2-x1).^2+(y2-y1).^2)*((x3-x1).^2+(y3-y1).^2)*((x3-x2).^2+(y3-y2).^2));
        otherwise  %no abs
            curvature(i) = 2*((x2-x1).*(y3-y1)-(x3-x1).*(y2-y1)) ./ ...
                sqrt(((x2-x1).^2+(y2-y1).^2)*((x3-x1).^2+(y3-y1).^2)*((x3-x2).^2+(y3-y2).^2));
    end
end

%--------------------------------------------------------------------------
%find the corresponding session file in another wavelength
function [match_dir] = find_session(curr_exp,curr_cond,curr_region,sesnum,file_loc)
match_dir = [];  %default output
for i = 1:size(file_loc,1)
    %parse and find
    sep_loc = find(file_loc{i}==filesep);  %find the seperator locations
    data_str = file_loc{i}(sep_loc(end)+1:end);  %take the text after the last seperator
    cond_tmp = file_loc{i}(sep_loc(end-2):sep_loc(end-1));  %find the current experimental condition,e.g., WT/KO or AM/PM
    region_tmp = file_loc{i}(sep_loc(end-1):sep_loc(end));   %find the current tissue region
    exp_tmp = file_loc{i}(sep_loc(end-3):sep_loc(end-2));      %find the current experiment
    %a match on all fronts is needed
    if strcmpi(sesnum,data_str(1))&&strcmpi(curr_cond,cond_tmp)&&strcmpi(curr_region,region_tmp)&&strcmpi(curr_exp,exp_tmp)
        match_dir = file_loc{i};
        break
    end
end

%--------------------------------------------------------------------------
function [stat_out,output,threshold,norm] = find_centroid_testrange_loco(dirname,type,type2,object,...
            threshold,norm,zfilter,stk,wavgroups)
%This is a high order script that allows you to run find_centroidvstk to
%look at a range of thresholds, for a range of normalizations.
%Synatax:   [] = find_centroid_testrange('dirname',array,'theshold',0.5,'type',2,'type2',1,'norm',1)
%Input:     'dirname' = directory where the images are, e.g., PSD95
%           'threshold' = if left off no threshold will be given, if given
%               the value should be between [0 1], and only maxima above
%               the desired level will be counted. Now by Default threshold
%               is 10% or 0.1 for localmax, centroids = 0.1 or 10%.  Now
%               the function will allow you to iterate through a number of
%               threshold you enter, essentially in the format:
%               (0.1,0.05...)  You should enter a range here for this
%               function. [0.05:0.05:0.5]  Meaning 0.05 to 0.5 by steps of
%               0.05.
%           type = 1(centroid), 2 (WeightedCentroid) or 0(local max),
%               default = 0(local max)
%           type2 = 1(Area), 2(Max Intensity), 3(Min Intensity, 4(Pixel #)
%                   default = 0(Mean Intensity) Note: only works with
%                   centroids
%           norm = normalize the images, essentially stretching the image
%                   data over the full bit range.  Only works in centroid
%                   mode.  Default = 1, or on, and will not affect the
%                   intensity  measurements, now matches threshold.  You
%                   should enter a range for this function, e.g., [0.99999,
%                   1, 0.999, 0.8...]
%           object = out put the images of the objects.  Default = 0(off)
%               Does not work with local max
%           zfilter = requires the objects to be in more than 1 z slice
%           Note: as this is setup currently, if there is a directory with
%           images that also contain directories, those images will not be
%           processed.  Images are only processed in end directories.
%           graph = graph the output.  Default = 1 (yes)
%           stk = You could enter the image stack if you want.
%           gnum = graph number, only used if you are graphing.
%           wavgroups = pregrouped data based on wavelength, if entered the
%               function will run in group mode.
%Output:    Output  = The matrix

%initiate
match_stk = [];

for j = 1:size(norm,2)  %each norm
    for i = 1:size(threshold,2)  %each threshold
        [pivots_tmp,cc_tmp,stk,stk_bw] = find_centroidvstk('dirname',dirname,'type',type,'type2',type2,'object',object,...
            'threshold',threshold(i),'norm',norm(j),'zfilter',zfilter,'save',0,'stk',stk);
        num_tmp = size(pivots_tmp{1},1);   %get the number of centroids
        %save number of output
        output(j,i) = num_tmp;
        %lets do some stats
        if num_tmp~=0  %no pivots
            vol = pivots_tmp{1}(:,5);   %pull the volume parameter
            stat_out(1,i,j) = mean(vol);
            %stat_out(2,i,j) = median(vol);
            %stat_out(3,i,j) = mode(vol);
        else
            stat_out(1,i,j) = 0;
            %stat_out(2,i,j) = 0;
            %stat_out(3,i,j) = 0;
        end
%         %lets look at the actual images generated
%         object_idx = cell2mat(struct2cell(regionprops(cc_tmp{1},'PixelIdxList'))');    %grab the indexs of all of the pixels
%         object_tmp = false(size(stk));    %stack holder
%         object_tmp(object_idx) = 1;     %create the image
%         object_img = object_tmp;   %store
        info = whos('stk');  %get the info for the stk
        imgclass = info.class;  %note you can get image size from this structure as well.
        switch imgclass
            case 'uint8'
                %object_lum = stk.*uint8(object_img);   %preserve the luminance.
                curr_stk = stk.*uint8(stk_bw);  %grab the image stack at current threshold - for corrlation analysis
            case 'uint16'
                %object_lum = stk.*uint16(object_img);   %preserve the luminance.
                curr_stk = stk.*uint16(stk_bw);  %grab the image stack at current threshold - for corrlation analysis
            case 'uint32'
                %object_lum = stk.*uint32(object_img);   %preserve the luminance.
                curr_stk = stk.*uint32(stk_bw);  %grab the image stack at current threshold - for corrlation analysis
            case 'double'
                %object_lum = stk.*double(object_img);   %preserve the luminance.
                curr_stk = stk.*double(stk_bw);  %grab the image stack at current threshold - for corrlation analysis
        end
%         %calculate the entropy of the images
%         %         stat_out(4,i,j) = entropy(object_img);
%         stat_out(5,i,j) = entropy(object_lum);
        %         figure('Name',[num2str(i),' ',num2str(stat_out(4,i,j)),' entropy of object binary @threshold: ',num2str(threshold(i))],'NumberTitle','off',...
        %             'OuterPosition',[200 500 800 500]);
        %         imshow(object_img(:,:,8));
        %         figure('Name',[num2str(i),' ',num2str(stat_out(5,i,j)),' entropy of object grayscale @threshold: ',num2str(threshold(i))],'NumberTitle','off',...
        %             'OuterPosition',[1500 500 800 500]);
        %         imshow(object_lum(:,:,8));
        %clear!
%         %calculate self correlation
%         for k = 2:size(curr_stk,3)  %go through each image
%             corr_tmp(k) = corr2(curr_stk(:,:,k-1),curr_stk(:,:,k));
%         end
%         stat_out(6,i,j) = nanmean(corr_tmp);  %output
%         %calculate accross threshold correlation
%         if i==1  %the first threshold
%             pre_stk = curr_stk;  %store and moveon
%             stat_out(7,i,j) = nan;  %empty
%         else
%             %now run the corr and calculate the mean core
%             for m = 1:size(curr_stk,3)
%                 corr_tmp(m) = corr2(pre_stk(:,:,m),curr_stk(:,:,m));
%             end
%             stat_out(7,i,j) = mean(corr_tmp);
%             pre_stk = curr_stk;
%         end
        %calculate accross threshold correlation to one singular threshold
        if i==1  %the first threshold
            for n = 1:size(stk,3)     %make the img Black & White
                pre_bw(:,:,n) = im2bw(stk(:,:,n),0.15);  %set compare threshold at 0.1
            end
            switch imgclass
                case 'uint8'
                    pre_stk = stk.*uint8(pre_bw);  %mask it
                case 'uint16'
                    pre_stk = stk.*uint16(pre_bw);  %mask it
                case 'uint32'
                    pre_stk = stk.*uint32(pre_bw);  %mask it
                case 'double'
                    pre_stk = stk.*double(pre_bw);  %mask it
            end
            stat_out(7,i,j) = nan;  %empty
        else
            %now run the corr and calculate the mean core
            for m = 1:size(curr_stk,3)
                corr_tmp(m) = corr2(pre_stk(:,:,m),curr_stk(:,:,m));
            end
            stat_out(7,i,j) = nanmean(corr_tmp);
            pre_stk = curr_stk;
        end
        %cross wavelength correlation
        if ~isempty(wavgroups)  %We need the wavgroups for this
            %parse the dirname
            sep_loc = find(dirname==filesep);  %find the seperator locations
            data_str = dirname(sep_loc(end)+1:end);  %take the text after the last seperator
            curr_cond = dirname(sep_loc(end-2):sep_loc(end-1));  %find the current experimental condition,e.g., WT/KO or AM/PM
            curr_region = dirname(sep_loc(end-1):sep_loc(end));   %find the current tissue region
            curr_exp = dirname(sep_loc(end-3):sep_loc(end-2));      %find the current experiment
            sesnum = data_str(1);  %get the image session session number  Format: 1 NR1 594 or 2 PSD95 594...
            wavnum = data_str(end-2:end);  %the last three are wavelength numbers
            %go forward if the wavelength is...
            if isempty(match_stk)
                match_dir = [];
                switch wavnum
                    case '647'
                        %grab the corresponding directory in 594
                        file_loc = wavgroups.lambda_594;
                    case '594'
                        file_loc = wavgroups.lambda_647;
                    otherwise
                        file_loc = [];
                end
                if ~isempty(file_loc)
                    [match_dir] = find_session(curr_exp,curr_cond,curr_region,sesnum,file_loc);
                    if ~isempty(match_dir)  %if there is a match
                        [match_stk] = openstk(match_dir);  %open the directory
                        match_bw = false(size(match_stk));
                        for l = 1:size(stk,3)     %make the img Black & White
                            try
                                match_bw(:,:,l) = im2bw(match_stk(:,:,l),0.1);  %set match threshold at 0.1
                            catch
                                keyboard
                            end
                        end
                        info = whos('match_stk');  %get the info for the stk
                        class_tmp = info.class;  %note you can get image size from this structure as well.
                        switch class_tmp
                            case 'uint8'
                                match_stk = match_stk.*uint8(match_bw);  %mask it
                            case 'uint16'
                                match_stk = match_stk.*uint16(match_bw);  %mask it
                            case 'uint32'
                                match_stk = match_stk.*uint32(match_bw);  %mask it
                            case 'double'
                                match_stk = match_stk.*double(match_bw);  %mask it
                        end
                    end
                end
            end
            if ~isempty(match_dir)  %must have a match to move forward
                %now run the corr and calculate the mean core
                for l = 1:size(curr_stk,3)
                    try
                        corr_tmp(l) = corr2(match_stk(:,:,l),curr_stk(:,:,l));
                    catch
                        keyboard
                    end
                end
                stat_out(8,i,j) = nanmean(corr_tmp);
            else
                %place holder
                stat_out(8,i,j) = nan;
            end
        else
            %place holder
            stat_out(8,i,j) = nan;
        end
        clear cc_tmp object_tmp object_img object_lum corr_tmp curr_stk corr_tmp
    end
    clear match_stk pre_stk pre_bw
end
%--------------------------------------------------------------------------
%subfunction to parse the image directories.
%takes an directory of image directory and groups them according to wave
%length
function [wav_groups] = groupwav(dirnames)
for i = 1:size(dirnames,1)
    spcidx = find(dirnames{i}==' ');  %find the spaces
    if ~isempty(spcidx)
        wavnum = dirnames{i}(spcidx(end)+1:end);  %the wave number
        try  %concatenate unless...
            wav_groups.(['lambda_',wavnum]) = vertcat(wav_groups.(['lambda_',wavnum]),dirnames(i));
        catch  %...The first time for this field
            wav_groups.(['lambda_',wavnum]) = dirnames(i);  %string numbers cannot be fields!  Thus the lamda_.
        end
    end
end
%--------------------------------------------------------------------------
function [pivots_out,output_dir] = find_centroidvstk_loco(varargin)
%This function is used to find the 3d centroid of a set of images in a
%directory, virutal stacks baby.  Loco version for specific output
%Synatax:   [pivots,centroid_img,lum_img] = find_centroidvstk('dirname',array,
%                               'theshold',0.5,'type',2,'type2',1,'norm',1)
%Input:     'dirname' = directory where the images are
%           'threshold' = if left off no threshold will be given, if given
%               the value should be between [0 1], and only maxima above
%               the desired level will be counted. Now by Default threshold
%               is 10% or 0.1 for localmax, centroids = 0.1 or 10%.  Now
%               the function will allow you to iterate through a number of
%               threshold you enter, essentially in the format:
%               (0.1,0.05...)
%           type = 1(centroid), 2 (WeightedCentroid) or 0(local max),
%               default = 0(local max)
%           type2 = 1(Area), 2(Max Intensity), 3(Min Intensity, 4(Pixel #)
%                   default = 0(Mean Intensity) Note: only works with
%                   centroids
%           norm = normalize the images, essentially stretching the image
%                   data over the full bit range.  Only works in centroid
%                   mode.  Default = 1, or on, and will not affect the
%                   intensity  measurements, now matches threshold.
%           object = out put the images of the objects.  Default = 0(off)
%               Does not work with local max
%           zfilter = requires the objects to be in more than 1 z slice
%           dirout = the output directory.  If not specified the output
%               directory will be in the root of the selected folder.
%           save = save out the images or not default = 1 (yes)
%           finetime = output directory contain hour and miniutes if
%               finetime is turned on. default = 0 (off)
%           savram = used in imnorm.  Check that function for info.
%Output:    pivots = a x by 4 array, where x y z define the pivots
%               and the brightness of the centroid is the 4th column.
%           output_dir = the output directory

[threshold,dirname,type,type2,norm,object,zfilter,dir_out,sav,finetime,savram,stk] = parse2(varargin);

%initiate labeling of matrix (happy now?!)
mxlabel = {'x','y','z'};
output_dir = {};

if isempty(stk)  %if no stacks are entered, get one
    %grab the filenames in the directory
    dir_struct = dir(dirname);  %grab the directory information
    idx = [dir_struct.isdir];   %grab all of the isdir numbers
    filenames = {dir_struct.name};   %grab the all of the names in the root
    filenames = filenames(~idx);  %don't want the directory names
    
    %Put the virtual stack into an actual volume
    [filenames] = findimgs(filenames);   %make sure we only take the image names
    info = imfinfo([dirname,filesep,filenames{1}]);     %grab the file info
    
    switch info.BitDepth    %what is the bit depth
        case 8
            imgclass = 'uint8';
        case 16
            imgclass = 'uint16';
        case 32
            imgclass = 'uint32';
        otherwise
            imgclass = 'double';
    end
    stk = zeros(info.Height,info.Width,size(filenames,2),imgclass); %generate holder variable
    
    h = waitbar(0,'Opening Image: ','position',[10 50 275 50]);     %initiate progress bar
    for i = 1:size(filenames,2)
        waitbar(i/size(filenames,2),h,['Opening Image: ',filenames{i}]);   %update progress
        stk(:,:,i) = imread([dirname,filesep,filenames{i}]);  %grab the image
    end
    close(h)
else  %stack is entered
    info = whos('stk');  %get the info for the stk
    imgclass = info.class;  %note you can get image size from this structure as well.
    i = 1;
end

stk_tmp = stk;        %make a copy of the stack to work on.
for m = 1:size(threshold,2)     %iterate for number of threshold
    %normalize, only if needed
    if m==1||norm(m)~=norm(m-1)
        if norm(m)==1  %normalize
            stk_tmp = imnorm(stk_tmp,[],[],savram);    %normalize the image so that the bit depth is maximized.
        elseif norm(m)>0 && norm(m)<1     %imadjust mode
            lo_hi = stretchlim(max(stk_tmp,[],3),[0 norm(m)]);   %generate the contrast range
            for l = 1:size(stk_tmp,3)   %enhance each image
                stk_tmp(:,:,l) = imadjust(stk_tmp(:,:,l),lo_hi,[]);
            end
        end
    end
    stk_bw = false(size(stk_tmp));
    for j = 1:size(stk,3)     %make the img Black & White
        stk_bw(:,:,j) = im2bw(stk_tmp(:,:,j),threshold(m));
    end
    stk_cc = bwconncomp(stk_bw);    %run the connected components analysis
    if zfilter(m)
        pixlst = regionprops(stk_cc,'PixelList');   %find the objects in the volume
        pixidxlst = regionprops(stk_cc,'PixelIdxList');     %get the pixel indexs for the objects
        pixlst = struct2cell(pixlst);    %convert to cell for simple useage
        for n = 1:size(pixlst,2)        %step through the array
            pixtmp(n) = size(unique(pixlst{1,n}(:,3)),1);   %we only care about whether the z is single slice or multi
        end
        rmvidx = find(pixtmp==1);   %find the single slice objects
        pixidxlst(rmvidx) = [];     %remove the single slice objects
        %now convert back to image for later calculations...admittedly I don't
        %need to do this, but to keep the rest of the function the same, I will
        %do this extra step.
        object_idx = cell2mat(struct2cell(pixidxlst)');    %grab the indexs of all of the pixels
        object_tmp = false(size(stk_bw));    %stack holder
        object_tmp(object_idx) = 1;     %create the image
        stk_cc = bwconncomp(object_tmp);   %store
    end
    %stk_bw = [];
    switch type
        case 1    %Centroid
            pivots = regionprops(stk_cc,'Centroid');  %find centroid
            %pivots_out_tmp = cat(1,pivots.Centroid);
        case 2    %WeightedCentroid
            pivots = regionprops(stk_cc,stk,'WeightedCentroid');
            %pivots_out_tmp = cat(1,pivots.WeightedCentroid);
    end
    if ~isempty(pivots)  %if pivots are empty, no objects thus finish
        lum = [];
        for o = 1:size(type2,2)
            switch type2(o)
                case 1      %area
                    lum_tmp = regionprops(stk_cc,'Area');
                    lum_tmp = struct2cell(lum_tmp);
                    lum = horzcat(lum,cell2mat(lum_tmp)');
                    if m==1 %do this once
                        mxlabel = [mxlabel 'Area']; %generate the label in order
                    end
                case 2      %max intensity
                    lum_tmp = regionprops(stk_cc,stk,'MaxIntensity');
                    lum_tmp = struct2cell(lum_tmp);
                    lum = horzcat(lum,cell2mat(lum_tmp)');
                    if m==1
                        mxlabel = [mxlabel 'MaxIntensity'];
                    end
                case 3      %min intensity
                    lum_tmp = regionprops(stk_cc,stk,'MinIntensity');
                    lum_tmp = struct2cell(lum_tmp);
                    lum = horzcat(lum,cell2mat(lum_tmp)');
                    if m==1
                        mxlabel = [mxlabel 'MinIntensity'];
                    end
                case 4      %Pixel Number
                    lum_tmp = regionprops(stk_cc,'PixelIdxList');
                    lum_tmp = struct2cell(lum_tmp);
                    for l = 1:size(lum_tmp,2); lum_tmp{l} = size(lum_tmp{l},1); end
                    lum = horzcat(lum,cell2mat(lum_tmp)');
                    if m==1
                        mxlabel = [mxlabel 'VoxelNumber'];
                    end
                case 5      %Intensity Integrated
                    lum_tmp = regionprops(stk_cc,stk,'PixelValue');
                    lum_tmp = struct2cell(lum_tmp);
                    for l = 1:size(lum_tmp,2); lum_tmp{l} = sum(lum_tmp{l}); end
                    lum = horzcat(lum,cell2mat(lum_tmp)');
                    if m==1
                        mxlabel = [mxlabel 'IntegratedIntensity'];
                    end
                otherwise
                    lum_tmp = regionprops(stk_cc,stk,'MeanIntensity');    %find the intensity of the regions
                    lum_tmp = struct2cell(lum_tmp);
                    lum = horzcat(lum,cell2mat(lum_tmp)');
                    if m==1
                        mxlabel = [mxlabel 'MeanIntensity'];
                    end
            end
            lum_tmp = [];
        end
        centroid_tmp = false(size(stk));  %create a logical array the same size as the original array
        %pivots is currently a structure, convert to cell then matrix
        pivots = cell2mat(struct2cell(pivots)');
        %now make the pivots output by adding lum
        pivots_out_tmp = [pivots lum];
        %round pivots to use as pixel indexs
        pivots = round(pivots);
        %now calculate the indexs for the centroid locations
        [x,y,z] = size(centroid_tmp);        %stack size
        pivots_idx = ((pivots(:,1)-1).*x)+pivots(:,2)+((x*y).*(pivots(:,3)-1));
        %modify img_tmp
        centroid_tmp(pivots_idx) = 1;
        
        centroid_img = centroid_tmp;
        pivots_tmp = pivots_out_tmp;
        if object   %if object is on
            object_idx = cell2mat(struct2cell(regionprops(stk_cc,'PixelIdxList'))');    %grab the indexs of all of the pixels
            object_tmp = false(size(stk));    %stack holder
            object_tmp(object_idx) = 1;     %create the image
            object_img = object_tmp;   %store
            switch imgclass
                case 'uint8'
                    object_lum = stk.*uint8(object_img);   %preserve the luminance.
                case 'uint16'
                    object_lum = stk.*uint16(object_img);   %preserve the luminance.
                case 'uint32'
                    object_lum = stk.*uint32(object_img);   %preserve the luminance.
                case 'double'
                    object_lum = stk.*double(object_img);   %preserve the luminance.
            end
            object_tmp = []; %clear!
        end
        %clear img_tmp pivots_out_tmp img_bw
        pivots_out_tmp = [];centroid_tmp = [];
        
        %Now pickup the brightness at the max for each centroid (single
        %point remember)
        switch imgclass
            case 'uint8'
                lum_img = stk.*uint8(centroid_img);
            case 'uint16'
                lum_img = stk.*uint16(centroid_img);
            case 'uint32'
                lum_img = stk.*uint32(centroid_img);
            case 'double'
                lum_img = stk.*double(centroid_img);
        end
        
        if sav  %save out only if we want to
            %dynamic path
            if isempty(dir_out)  %not selected output directory
                sep_loc = strfind(dirname,filesep);
                dir_out = dirname(1:sep_loc(end));  %the root of the selected directory.
            end
            if finetime
                switch type
                    case 0
                        dir_path{m} = ['LocalMaxs',datestr(now,'yyyymmdd.HHMM'),'_t',num2str(threshold(m)),'z',num2str(zfilter(m)),'n',num2str(norm(m)),filesep];     %where to save it
                        object = 0;     %force this, because local max has no object
                    case 1
                        dir_path{m} = ['Centroids',datestr(now,'yyyymmdd.HHMM'),'_t',num2str(threshold(m)),'z',num2str(zfilter(m)),'n',num2str(norm(m)),filesep];     %where to save it
                    case 2
                        dir_path{m} = ['WeightedCentroids',datestr(now,'yyyymmdd.HHMM'),'_t',num2str(threshold(m)),'z',num2str(zfilter(m)),'n',num2str(norm(m)),filesep];     %where to save it
                end
            else
                switch type
                    case 0
                        dir_path{m} = ['LocalMaxs',datestr(now,'yyyymmdd'),'_t',num2str(threshold(m)),'z',num2str(zfilter(m)),'n',num2str(norm(m)),filesep];     %where to save it
                        object = 0;     %force this, because local max has no object
                    case 1
                        dir_path{m} = ['Centroids',datestr(now,'yyyymmdd'),'_t',num2str(threshold(m)),'z',num2str(zfilter(m)),'n',num2str(norm(m)),filesep];     %where to save it
                    case 2
                        dir_path{m} = ['WeightedCentroids',datestr(now,'yyyymmdd'),'_t',num2str(threshold(m)),'z',num2str(zfilter(m)),'n',num2str(norm(m)),filesep];     %where to save it
                end
            end
            warning('OFF')
            mkdir(dir_out,dir_path{m});      %create output directory
            mkdir(dir_out,[dir_path{m},'img_data',filesep]);     %create a directory where the file information is stored
            %now save the files
            stridx = strfind(dirname,filesep);  %create a label for the file
            stklabel = dirname(stridx(end)+1:end);
            
            mkdir([dir_out,dir_path{m}],stklabel);
            xls_exp = dataset({pivots_tmp,mxlabel{:}});               %to export as a XLS need to make the data a dataset.
            export(xls_exp,'File',[[dir_out,dir_path{m}],[stklabel,'.',num2str(size(pivots_tmp,1)),'.csv']],'delimiter',',')
            mkdir([dir_out,dir_path{m},stklabel],['lum_imgs',filesep]);
            stk2tiff(lum_img,stklabel,[dir_out,dir_path{m},stklabel,filesep,'lum_imgs',filesep]);
            mkdir([dir_out,dir_path{m},stklabel],['centroid_imgs',filesep]);
            stk2tiff(centroid_img,stklabel,[dir_out,dir_path{m},stklabel,filesep,'centroid_imgs',filesep]);
            save([[dir_out,dir_path{m}],[stklabel,'_cc','.mat']],'stk_cc');
            if object   %save out the object stacks
                mkdir([dir_out,dir_path{m},stklabel],['objects_lum',filesep]);
                stk2tiff(object_lum,stklabel,[dir_out,dir_path{m},stklabel,filesep,'objects_lum',filesep]);
                mkdir([dir_out,dir_path{m},stklabel],['objects',filesep]);
                stk2tiff(object_img,stklabel,[dir_out,dir_path{m},stklabel,filesep,'objects',filesep]);
            end
            %save some info about the image
            img_data = size(stk);
            img_data = dataset(img_data);
            export(img_data,'File',[[dir_out,dir_path{m},'img_data',filesep],stklabel,'_img_data.csv'],'delimiter',',')
            if i==1
                %for historical reasons I am going to do this, just one more file no big
                export(img_data,'File',[[dir_out,dir_path{m},'img_data',filesep],'img_data.csv'],'delimiter',',')
            end
            warning('ON')
            output_dir{m} = [dir_out,dir_path{m}];
        end
    else
        %no objects - output empty
        pivots_tmp = [];
        stk_cc = [];
    end
    %output pivots
    pivots_out{m} = pivots_tmp;
    %output cc structure
    cc_out{m} = stk_cc;
    %clear data
    local_max_tmp = []; lum_tmp = []; xls_exp = [];centroid_img = [];lum_img = [];object_img = [];object_lum = [];pivots_tmp = [];stk_cc=[];
end

%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [threshold,dirname,type,type2,norm,object,zfilter,dir_out,sav,finetime,savram,stk] = parse2(input)

threshold = [];
type = 0;
type2 = 0;      %default is 0; now you can define the accessory data, acquired instead of Luminance
norm = 1;
object = 0;
dirname = [];
zfilter = 0;
dir_out = [];
sav = 1;
finetime = 0;
savram = 0;
stk = [];

%Parse the input
if ~isempty(input)
    for i = 1:2:size(input,2)
        if ischar(input{1,i});
            switch input{1,i}
                case 'dirname'
                    dirname = input{1,i+1};
                case 'threshold'
                    threshold = input{1,i+1};
                case 'type'
                    type = input{1,i+1};
                case 'type2'
                    type2 = input{1,i+1};
                case 'norm'
                    norm = input{1,i+1};
                case 'object'
                    object = input{1,i+1};
                case 'zfilter'
                    zfilter = input{1,i+1};
                case 'dirout'
                    dir_out = input{1,i+1};
                case 'save'
                    sav = input{1,i+1};
                case 'finetime'
                    finetime = input{1,i+1};
                case 'savram'
                    savram = input{1,i+1};
                case 'stk'
                    stk = input{1,i+1};
                otherwise
                    warning(['Your input ',input{1,i},' is not recognized.']);
            end
        else
            error(['The parameters you entered is incorrect.  Please check help.']);
        end
    end
end

%Get the location of the images you want to open
if isempty(dirname)&&isempty(stk)
    dirname = uigetdir2('','Directory where your images are');    %get the directory
end
%set threshold, if the user have not already.
if isempty(threshold)
    switch type
        case 0   %default local max
            threshold = 0.1;
        otherwise  %centroid or weighted centroid
            threshold = 0.1;
    end
end
%make sure treshold and norm line up
if size(threshold,2)~=size(norm,2)   %align zfilter to threshold
    diff = size(threshold,2)-size(norm,2);   %the difference
    norm = horzcat(norm,ones(1,diff));   %pad with default
end
%make sure treshold and zfilter line up
if size(threshold,2)~=size(zfilter,2)   %align zfilter to threshold
    diff = size(threshold,2)-size(zfilter,2);   %the difference
    zfilter = horzcat(zfilter,zeros(1,diff));   %pad with default
end
%we are going to make sure the dirout has a filesep at the end.  You should
%also make sure the dirout is not a cell, if that could happen, not
%necessary now.
if ~isempty(dir_out)
    sep_loc = strfind(dir_out,filesep);     %find all of the filesep locations
    if sep_loc(end)~=size(dir_out,2)    %if the end does not have a filesep
        dir_out = [dir_out,filesep];    %add it
    end
end